"""
Axiom Streaming Client - GUI for Distributed Hebbian Learning

Combines BOINC authentication with streaming Hebbian training.
'Neurons that fire together, wire together'
"""

import os
import sys
import time
import traceback

# === STARTUP LOGGING ===
# Log to file in same directory as script for debugging startup issues
_SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
_STARTUP_LOG = os.path.join(_SCRIPT_DIR, "startup_log.txt")

def _log_startup(msg):
    """Write startup message to log file."""
    try:
        with open(_STARTUP_LOG, "a") as f:
            f.write(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] {msg}\n")
    except:
        pass

_log_startup(f"=== STARTUP BEGIN ===")
_log_startup(f"Python version: {sys.version}")
_log_startup(f"Platform: {sys.platform}")
_log_startup(f"Script: {__file__}")
_log_startup(f"Working dir: {os.getcwd()}")
import json
import hashlib
import threading
import subprocess
import multiprocessing as mp
import xml.etree.ElementTree as ET
from pathlib import Path
from dataclasses import dataclass, field
from urllib.request import urlopen, Request
from urllib.parse import urlencode
from urllib.error import URLError, HTTPError
import tkinter as tk
from tkinter import ttk, messagebox

# Try to import psutil for system monitoring
try:
    import psutil
    HAS_PSUTIL = True
except ImportError:
    HAS_PSUTIL = False

# Try to import pystray for system tray
try:
    import pystray
    from PIL import Image, ImageDraw
    HAS_TRAY = True
except ImportError:
    HAS_TRAY = False

# Configuration
PROJECT_URL = "https://axiom.heliex.net/"
PROJECT_NAME = "Axiom"
APP_NAME = "Axiom Streaming Client"
APP_VERSION = "2.0.3"
DATA_DIR = Path(os.environ.get('APPDATA', Path.home())) / "AxiomClient"
CONFIG_FILE = DATA_DIR / "config.json"
STREAMING_SERVER = "ws://65.21.196.61:8765"
CONTRIBUTE_DIR = Path(os.environ.get('USERPROFILE', Path.home())) / "Axiom" / "contribute"
UPDATE_URL = "https://axiom.heliex.net/client/"  # Where version.json and files are hosted

# Platform info
PLATFORM = "windows_x86_64"


@dataclass
class StreamingConfig:
    """Configuration for streaming training"""
    num_workers: int = 4
    expert_type: str = "transformer"
    d_model: int = 768
    n_heads: int = 12
    expert_hidden: int = 3072
    expert_layers: int = 6
    hebbian_lr: float = 0.001
    hebbian_decay: float = 0.999
    sync_interval: int = 1000
    gossip_alpha: float = 0.15
    memory_threshold_pct: float = 85.0
    max_storage_pct: float = 0.0  # Stop if disk usage exceeds this % (0 = disabled)
    min_storage_gb: float = 3.0   # Stop if free space falls below this GB (default 3GB)
    # CPU settings
    cpu_mode: str = "adaptive"  # "fixed" or "adaptive"
    cpu_fixed_pct: float = 50.0  # For fixed mode: use this % of CPU
    cpu_backoff_threshold: float = 70.0  # For adaptive mode: back off when system CPU > this
    # Schedule settings
    schedule_enabled: bool = False
    schedule_days: list = field(default_factory=lambda: [0, 1, 2, 3, 4, 5, 6])  # 0=Mon, 6=Sun
    schedule_start_hour: int = 0
    schedule_start_minute: int = 0
    schedule_end_hour: int = 23
    schedule_end_minute: int = 59
    # Appearance
    theme: str = "dark"  # "dark" or "light"
    # Warnings
    hide_compressed_warning: bool = False
    # State
    was_running: bool = False


class AxiomConfig:
    """Persistent configuration including BOINC auth"""
    def __init__(self):
        self.project_url = PROJECT_URL
        self.authenticator = ""
        self.email = ""
        self.userid = 0
        self.username = ""
        self.hostid = 0
        self.total_credit = 0.0
        self.expavg_credit = 0.0
        self.streaming = StreamingConfig()
        self.load()

    def load(self):
        if CONFIG_FILE.exists():
            try:
                with open(CONFIG_FILE, 'r') as f:
                    data = json.load(f)
                    self.authenticator = data.get('authenticator', '')
                    self.email = data.get('email', '')
                    self.userid = data.get('userid', 0)
                    self.username = data.get('username', '')
                    self.hostid = data.get('hostid', 0)
                    self.total_credit = data.get('total_credit', 0.0)
                    self.expavg_credit = data.get('expavg_credit', 0.0)
                    # Load streaming config
                    sc = data.get('streaming', {})
                    self.streaming.num_workers = sc.get('num_workers', 4)
                    self.streaming.expert_type = sc.get('expert_type', 'transformer')
                    self.streaming.sync_interval = sc.get('sync_interval', 1000)
                    self.streaming.memory_threshold_pct = sc.get('memory_threshold_pct', 85.0)
                    self.streaming.max_storage_pct = sc.get('max_storage_pct', 0.0)
                    self.streaming.min_storage_gb = sc.get('min_storage_gb', 3.0)
                    # CPU settings
                    self.streaming.cpu_mode = sc.get('cpu_mode', 'adaptive')
                    self.streaming.cpu_fixed_pct = sc.get('cpu_fixed_pct', 50.0)
                    self.streaming.cpu_backoff_threshold = sc.get('cpu_backoff_threshold', 70.0)
                    # Schedule settings
                    self.streaming.schedule_enabled = sc.get('schedule_enabled', False)
                    self.streaming.schedule_days = sc.get('schedule_days', [0, 1, 2, 3, 4, 5, 6])
                    self.streaming.schedule_start_hour = sc.get('schedule_start_hour', 0)
                    self.streaming.schedule_start_minute = sc.get('schedule_start_minute', 0)
                    self.streaming.schedule_end_hour = sc.get('schedule_end_hour', 23)
                    self.streaming.schedule_end_minute = sc.get('schedule_end_minute', 59)
                    self.streaming.theme = sc.get('theme', 'dark')
                    self.streaming.hide_compressed_warning = sc.get('hide_compressed_warning', False)
                    self.streaming.was_running = sc.get('was_running', False)
            except Exception as e:
                print(f"Error loading config: {e}")

    def save(self):
        DATA_DIR.mkdir(parents=True, exist_ok=True)
        with open(CONFIG_FILE, 'w') as f:
            json.dump({
                'authenticator': self.authenticator,
                'email': self.email,
                'userid': self.userid,
                'username': self.username,
                'hostid': self.hostid,
                'total_credit': self.total_credit,
                'expavg_credit': self.expavg_credit,
                'streaming': {
                    'num_workers': self.streaming.num_workers,
                    'expert_type': self.streaming.expert_type,
                    'sync_interval': self.streaming.sync_interval,
                    'memory_threshold_pct': self.streaming.memory_threshold_pct,
                    'max_storage_pct': self.streaming.max_storage_pct,
                    'min_storage_gb': self.streaming.min_storage_gb,
                    'cpu_mode': self.streaming.cpu_mode,
                    'cpu_fixed_pct': self.streaming.cpu_fixed_pct,
                    'cpu_backoff_threshold': self.streaming.cpu_backoff_threshold,
                    'schedule_enabled': self.streaming.schedule_enabled,
                    'schedule_days': self.streaming.schedule_days,
                    'schedule_start_hour': self.streaming.schedule_start_hour,
                    'schedule_start_minute': self.streaming.schedule_start_minute,
                    'schedule_end_hour': self.streaming.schedule_end_hour,
                    'schedule_end_minute': self.streaming.schedule_end_minute,
                    'theme': self.streaming.theme,
                    'hide_compressed_warning': self.streaming.hide_compressed_warning,
                    'was_running': self.streaming.was_running,
                }
            }, f, indent=2)

    def logout(self):
        self.authenticator = ""
        self.email = ""
        self.userid = 0
        self.username = ""
        self.hostid = 0
        self.save()


class AccountManager:
    """Handles BOINC account operations"""

    def __init__(self, config: AxiomConfig):
        self.config = config

    def _make_request(self, endpoint, params):
        url = self.config.project_url.rstrip('/') + '/' + endpoint + '?' + urlencode(params)
        try:
            req = Request(url, headers={'User-Agent': f'{APP_NAME}/{APP_VERSION}'})
            with urlopen(req, timeout=30) as resp:
                return resp.read().decode('utf-8')
        except Exception as e:
            return f"<error_msg>{str(e)}</error_msg>"

    def _make_post_request(self, endpoint, params):
        url = self.config.project_url.rstrip('/') + '/' + endpoint
        try:
            data = urlencode(params).encode('utf-8')
            req = Request(url, data=data, headers={'User-Agent': f'{APP_NAME}/{APP_VERSION}'})
            with urlopen(req, timeout=30) as resp:
                return resp.read().decode('utf-8')
        except Exception as e:
            return f"<error_msg>{str(e)}</error_msg>"

    def login(self, email, password):
        passwd_hash = hashlib.md5((password + email.lower()).encode()).hexdigest()
        response = self._make_request('lookup_account.php', {
            'email_addr': email,
            'passwd_hash': passwd_hash,
            'get_opaque_auth': '1'
        })

        try:
            root = ET.fromstring(response)
        except ET.ParseError:
            return False, "Invalid response from server"

        error = root.find('error_msg')
        if error is not None:
            return False, error.text

        auth = root.find('authenticator')
        if auth is not None and auth.text:
            self.config.authenticator = auth.text.strip()
            self.config.email = email
            self.config.save()
            # Fetch credit immediately after login
            self.fetch_credit()
            return True, "Login successful"

        return False, "Login failed - no authenticator returned"

    def register(self, email, username, password):
        passwd_hash = hashlib.md5((password + email.lower()).encode()).hexdigest()
        response = self._make_request('create_account.php', {
            'email_addr': email,
            'passwd_hash': passwd_hash,
            'user_name': username
        })

        try:
            root = ET.fromstring(response)
        except ET.ParseError:
            return False, "Invalid response from server"

        error = root.find('error_msg')
        if error is not None:
            return False, error.text

        auth = root.find('authenticator')
        if auth is not None and auth.text:
            self.config.authenticator = auth.text.strip()
            self.config.email = email
            self.config.username = username
            self.config.save()
            return True, "Account created successfully"

        return False, "Registration failed"

    def forgot_password(self, email):
        response = self._make_post_request('mail_passwd.php', {'email_addr': email})
        # BOINC returns HTML, not XML for this endpoint
        if 'Email sent' in response or 'Instructions' in response:
            return True, "If the email exists, a reset link has been sent."
        elif 'No such user' in response:
            return True, "If the email exists, a reset link has been sent."  # Don't reveal if account exists
        elif 'disabled' in response:
            return False, "This account has been disabled."
        elif 'error_msg' in response:
            return False, "Failed to send reset email."
        return True, "If the email exists, a reset link has been sent."

    def fetch_credit(self):
        """Fetch current credit from server using authenticator."""
        if not self.config.authenticator:
            self.last_error = "No authenticator"
            return False, 0.0, 0.0

        response = self._make_request('show_user.php', {
            'auth': self.config.authenticator,
            'format': 'xml'
        })

        try:
            root = ET.fromstring(response)
        except ET.ParseError:
            self.last_error = f"Parse error: {response[:100]}"
            return False, 0.0, 0.0

        error = root.find('error_msg')
        if error is not None:
            self.last_error = f"Server error: {error.text}"
            return False, 0.0, 0.0

        total_credit = 0.0
        expavg_credit = 0.0

        tc = root.find('total_credit')
        if tc is not None and tc.text:
            try:
                total_credit = float(tc.text)
            except ValueError:
                pass

        ec = root.find('expavg_credit')
        if ec is not None and ec.text:
            try:
                expavg_credit = float(ec.text)
            except ValueError:
                pass

        # Update config with latest credit values
        self.config.total_credit = total_credit
        self.config.expavg_credit = expavg_credit

        return True, total_credit, expavg_credit


class UpdateManager:
    """Handles checking for and applying updates"""

    def __init__(self, on_log=None):
        self.on_log = on_log
        self.latest_version = None
        self.update_available = False
        self.changelog = ""
        self.files_to_update = []

    def check_for_updates(self):
        """Check the server for available updates. Returns (has_update, version, changelog)"""
        try:
            url = UPDATE_URL + "version.json"
            req = Request(url, headers={'User-Agent': f'{APP_NAME}/{APP_VERSION}'})
            with urlopen(req, timeout=10) as resp:
                data = json.loads(resp.read().decode('utf-8'))

            self.latest_version = data.get('version', APP_VERSION)
            self.changelog = data.get('changelog', '')
            self.files_to_update = data.get('files', [
                'axiom_streaming_ui.py',
                'axiom_streaming_client.py',
                'simple_ml.py'
            ])

            # Compare versions
            self.update_available = self._compare_versions(APP_VERSION, self.latest_version)

            return self.update_available, self.latest_version, self.changelog

        except Exception as e:
            if self.on_log:
                self.on_log(f"Update check failed: {e}", 'warning')
            return False, APP_VERSION, ""

    def _compare_versions(self, current, latest):
        """Compare version strings. Returns True if latest > current"""
        try:
            current_parts = [int(x) for x in current.split('.')]
            latest_parts = [int(x) for x in latest.split('.')]
            return latest_parts > current_parts
        except:
            return False

    def download_update(self, target_dir):
        """Download update files to target directory. Returns (success, message)"""
        if not self.update_available:
            return False, "No update available"

        try:
            downloaded = []
            for filename in self.files_to_update:
                url = UPDATE_URL + filename
                if self.on_log:
                    self.on_log(f"Downloading {filename}...", 'info')

                req = Request(url, headers={'User-Agent': f'{APP_NAME}/{APP_VERSION}'})
                with urlopen(req, timeout=30) as resp:
                    content = resp.read()

                # Save to target directory
                filepath = Path(target_dir) / filename
                with open(filepath, 'wb') as f:
                    f.write(content)

                downloaded.append(filename)

            return True, f"Downloaded {len(downloaded)} files"

        except Exception as e:
            return False, f"Download failed: {e}"


class WorkerStats:
    """Statistics for a single worker"""
    def __init__(self, worker_id: int):
        self.worker_id = worker_id
        self.expert_idx = None  # Assigned by coordinator
        self.pid = 0
        self.samples = 0
        self.syncs = 0
        self.samples_since_sync = 0  # Progress toward next sync
        self.rate = 0.0
        self.connected = False
        self.state = "starting"  # starting, waiting, training, syncing


class StreamingManager:
    """Manages the streaming training workers"""

    def __init__(self, config: AxiomConfig, on_update=None, on_log=None):
        self.config = config
        self.on_update = on_update
        self.on_log = on_log
        self.running = False
        self.workers: dict[int, WorkerStats] = {}
        self.process = None
        self.output_thread = None
        self.total_samples = 0
        self.total_syncs = 0
        self.cpu_percent = 0.0
        self.memory_percent = 0.0
        self.memory_gb = 0.0
        self.memory_total_gb = 0.0
        self.active_worker_count = 0

    def start(self):
        if self.running:
            return

        self.running = True
        self.workers.clear()
        self.total_samples = 0
        self.total_syncs = 0
        self.active_worker_count = 0  # Reset - will be updated if workers are killed

        # Initialize worker stats
        for i in range(self.config.streaming.num_workers):
            self.workers[i] = WorkerStats(i)

        # Find the streaming client script
        script_path = Path(__file__).parent / "axiom_streaming_client.py"
        if not script_path.exists():
            script_path = Path(__file__).parent / "axiom_streaming_client.py"

        if not script_path.exists():
            if self.on_log:
                self.on_log("Error: axiom_streaming_client.py not found", 'error')
            self.running = False
            return

        # Start the streaming client as a subprocess
        env = os.environ.copy()
        env['PYTHONUNBUFFERED'] = '1'

        cmd = [
            sys.executable, str(script_path),
            '--parallel', str(self.config.streaming.num_workers),
            '--expert-type', self.config.streaming.expert_type,
            '--sync-interval', str(self.config.streaming.sync_interval),
            '--memory-threshold', str(self.config.streaming.memory_threshold_pct),
            '--cpu-mode', self.config.streaming.cpu_mode,
            '--cpu-fixed-pct', str(self.config.streaming.cpu_fixed_pct),
            '--cpu-backoff-threshold', str(self.config.streaming.cpu_backoff_threshold),
        ]

        # Add authenticator for credit tracking
        if self.config.authenticator:
            cmd.extend(['--authenticator', self.config.authenticator])
            if self.on_log:
                self.on_log(f"Using authenticator: {self.config.authenticator[:8]}...", 'info')
        else:
            if self.on_log:
                self.on_log("Warning: No authenticator - credits won't be tracked!", 'warning')

        if self.on_log:
            self.on_log(f"Starting {self.config.streaming.num_workers} workers...", 'info')

        try:
            self.process = subprocess.Popen(
                cmd,
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT,
                env=env,
                creationflags=subprocess.CREATE_NO_WINDOW,
                bufsize=1,
                universal_newlines=True
            )

            # Start output reading thread
            self.output_thread = threading.Thread(target=self._read_output, daemon=True)
            self.output_thread.start()

            if self.on_log:
                self.on_log(f"Streaming client started (PID: {self.process.pid})", 'success')

        except Exception as e:
            if self.on_log:
                self.on_log(f"Failed to start: {e}", 'error')
            self.running = False

    def _read_output(self):
        """Read output from the streaming client process"""
        try:
            for line in self.process.stdout:
                line = line.strip()
                if not line:
                    continue

                # Parse the output
                self._parse_line(line)

                # Log certain messages
                if self.on_log:
                    if any(x in line for x in ['Error', 'error', 'Connected', 'Starting', 'initialized', 'Skipping', 'Warning']):
                        level = 'warning' if ('Skipping' in line or 'Warning' in line) else 'info'
                        self.on_log(line, level)
                    elif '[Sync]' in line and 'Sync #' in line:
                        # Log sync completions
                        self.on_log(line, 'success')
                    elif 'Assigned expert' in line:
                        # Log expert assignments
                        self.on_log(line, 'info')

        except Exception as e:
            if self.on_log:
                self.on_log(f"Output reader error: {e}", 'error')

        self.running = False
        if self.on_update:
            self.on_update()

    def _parse_line(self, line: str):
        """Parse a line of output from the streaming client"""
        sync_interval = self.config.streaming.sync_interval

        # Parse worker-specific messages
        # [Worker-X] Assigned expert Y
        if '[Worker-' in line:
            try:
                idx = int(line.split('[Worker-')[1].split(']')[0])
                if idx in self.workers:
                    if 'Starting' in line:
                        self.workers[idx].state = 'starting'
                    elif 'Waiting' in line:
                        self.workers[idx].state = 'waiting'
                    elif 'Assigned expert' in line:
                        # [Worker-X] Assigned expert Y
                        expert_idx = int(line.split('expert')[1].strip())
                        self.workers[idx].expert_idx = expert_idx
                        self.workers[idx].state = 'training'
                    elif 'Expert:' in line and 'Progress:' in line:
                        # [Worker-X] Expert: E | Progress: P/T (from main process)
                        parts = line.split('|')
                        expert = int(parts[0].split('Expert:')[1].strip())
                        progress = int(parts[1].split(':')[1].strip().split('/')[0])
                        self.workers[idx].expert_idx = expert
                        self.workers[idx].samples_since_sync = progress
                        self.workers[idx].state = 'training'
            except:
                pass

        # [Sync] Worker-X Expert-Y: Sync #Z complete
        if '[Sync] Worker-' in line and 'Sync #' in line:
            try:
                worker_idx = int(line.split('Worker-')[1].split()[0])
                if worker_idx in self.workers:
                    # Don't reset samples_since_sync here - worker output is authoritative
                    self.workers[worker_idx].state = 'syncing'
            except:
                pass

        # [Monitor] CPU: X% | RAM: Y% (Z/T GB) | Samples: N | Syncs: S
        if '[Monitor]' in line:
            try:
                parts = line.replace('[Monitor]', '').split('|')
                self.cpu_percent = float(parts[0].split(':')[1].strip().replace('%', ''))
                ram_part = parts[1].split(':')[1].strip()
                self.memory_percent = float(ram_part.split('%')[0])
                mem_values = ram_part.split('(')[1].replace(')', '').replace('GB', '').split('/')
                self.memory_gb = float(mem_values[0])
                self.memory_total_gb = float(mem_values[1])
                # Parse samples and syncs
                for part in parts[2:]:
                    if 'Samples:' in part:
                        self.total_samples = int(part.split(':')[1].strip().replace(',', ''))
                    elif 'Syncs:' in part:
                        self.total_syncs = int(part.split(':')[1].strip())
            except:
                pass

        # [WORKERS] Active: N
        if '[WORKERS] Active:' in line:
            try:
                self.active_worker_count = int(line.split(':')[1].strip())
                if self.on_log:
                    self.on_log(f"Workers reduced to {self.active_worker_count} (memory pressure)", 'warning')
            except:
                pass

        # [WeightSync] Connected!
        if '[WeightSync] Connected' in line:
            # Mark all workers as connected
            for w in self.workers.values():
                w.connected = True

        # Update callback
        if self.on_update:
            self.on_update()

    def stop(self):
        if not self.running:
            return

        self.running = False

        if self.on_log:
            self.on_log("Stopping workers...", 'warning')

        # Kill the process tree
        if self.process:
            try:
                # Kill all python processes spawned by the streaming client
                subprocess.run(
                    ['taskkill', '/F', '/T', '/PID', str(self.process.pid)],
                    capture_output=True,
                    timeout=10,
                    creationflags=subprocess.CREATE_NO_WINDOW
                )
            except:
                pass

            try:
                self.process.terminate()
                self.process.wait(timeout=5)
            except:
                try:
                    self.process.kill()
                except:
                    pass

        self.process = None

        if self.on_log:
            self.on_log("Workers stopped", 'info')

        if self.on_update:
            self.on_update()

    def get_status_summary(self) -> str:
        if not self.running:
            return "Stopped"

        connected = sum(1 for w in self.workers.values() if w.connected)
        total = len(self.workers)

        # Show active worker count if reduced due to memory pressure
        if self.active_worker_count > 0 and self.active_worker_count < total:
            return f"Training ({self.active_worker_count}/{total} workers)"

        if connected < total:
            return f"Connecting ({connected}/{total})"
        return f"Training ({total} workers)"


class AxiomStreamingGUI:
    """Main GUI for Axiom Streaming Client"""

    def __init__(self):
        _log_startup("AxiomStreamingGUI.__init__() starting")
        self.config = AxiomConfig()
        _log_startup("AxiomConfig loaded")

        # Kill any existing processes
        self._kill_existing()
        _log_startup("_kill_existing() done")

        self.root = tk.Tk()
        _log_startup("tk.Tk() created")
        self.root.title(f"{APP_NAME} v{APP_VERSION}")
        self.root.geometry("900x750")
        self.root.minsize(800, 650)

        self._setup_ui()

        # Initialize managers
        self.account_manager = AccountManager(self.config)
        self.streaming_manager = StreamingManager(
            self.config,
            on_update=self._schedule_refresh,
            on_log=self._thread_safe_log
        )
        self.update_manager = UpdateManager(on_log=self._thread_safe_log)

        # Setup system tray
        self.tray_icon = None
        if HAS_TRAY:
            self._setup_tray()

        # Credit refresh tracking
        self.last_credit_fetch = 0

        # Suspend state tracking
        self.training_suspended = False

        self._update_status()

        # Apply saved theme
        if self.config.streaming.theme == "light":
            self.theme_var.set("light")
            self._apply_theme()

        # Show login if not authenticated
        if not self.config.authenticator:
            self.root.after(100, self.show_login)

        # Check for updates on startup (after 3 seconds)
        self.root.after(3000, self._check_updates_background)

    def _kill_existing(self):
        """Kill any existing Axiom client and streaming processes.

        This is best-effort only - if it fails, we continue anyway.
        Uses PowerShell instead of deprecated wmic (removed in Win11/Python 3.13).
        """
        _log_startup("_kill_existing() called")
        try:
            current_pid = os.getpid()
            parent_pid = os.getppid()  # Don't kill parent (venv launcher on Windows)
            _log_startup(f"Current PID: {current_pid}, Parent PID: {parent_pid}")

            # Setup to hide console window (compatible with all Python versions)
            startupinfo = None
            creationflags = 0
            if sys.platform == 'win32':
                startupinfo = subprocess.STARTUPINFO()
                startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW
                startupinfo.wShowWindow = 0  # SW_HIDE
                creationflags = getattr(subprocess, 'CREATE_NO_WINDOW', 0x08000000)

            # Try PowerShell method (works on Windows 10/11, doesn't hang like wmic)
            _log_startup("Running PowerShell to find existing processes...")
            ps_cmd = (
                "Get-CimInstance Win32_Process | "
                "Where-Object { $_.Name -match 'python' -and $_.CommandLine -match 'axiom_streaming' } | "
                "Select-Object ProcessId | ForEach-Object { $_.ProcessId }"
            )
            result = subprocess.run(
                ['powershell', '-NoProfile', '-Command', ps_cmd],
                capture_output=True, text=True, timeout=5,
                startupinfo=startupinfo, creationflags=creationflags
            )
            _log_startup(f"PowerShell result: stdout='{result.stdout.strip()}', stderr='{result.stderr.strip()}'")

            for line in result.stdout.strip().splitlines():
                try:
                    pid = int(line.strip())
                    if pid != current_pid and pid != parent_pid and pid > 0:
                        _log_startup(f"Killing PID {pid} (not current={current_pid}, not parent={parent_pid})")
                        subprocess.run(
                            ['taskkill', '/F', '/PID', str(pid)],
                            capture_output=True, timeout=3,
                            startupinfo=startupinfo, creationflags=creationflags
                        )
                except (ValueError, OSError, subprocess.SubprocessError) as e:
                    _log_startup(f"Error killing PID: {e}")
                    pass
            _log_startup("_kill_existing() completed successfully")
        except Exception as e:
            _log_startup(f"_kill_existing() failed: {e}\n{traceback.format_exc()}")
            pass  # Not critical - continue startup even if this fails

    def _setup_ui(self):
        # Modern dark theme colors
        BG_DARK = '#0d1117'       # GitHub dark background
        BG_CARD = '#161b22'       # Card/panel background
        BG_INPUT = '#21262d'      # Input fields
        BG_BUTTON = '#238636'     # Primary button (green)
        BG_BUTTON_SEC = '#21262d' # Secondary button
        BORDER = '#30363d'        # Borders
        FG_PRIMARY = '#f0f6fc'    # Main text
        FG_SECONDARY = '#8b949e'  # Secondary text
        FG_ACCENT = '#58a6ff'     # Links/accents (blue)
        SUCCESS = '#3fb950'       # Green
        ERROR = '#f85149'         # Red
        WARNING = '#d29922'       # Orange/yellow

        # Set root background
        self.root.configure(bg=BG_DARK)

        style = ttk.Style()
        style.theme_use('clam')

        # Configure dark theme for all widgets
        style.configure('.', background=BG_DARK, foreground=FG_PRIMARY, fieldbackground=BG_INPUT,
                        bordercolor=BORDER, lightcolor=BG_CARD, darkcolor=BG_DARK)
        style.configure('TFrame', background=BG_DARK)
        style.configure('TLabelframe', background=BG_DARK, foreground=FG_PRIMARY, bordercolor=BORDER)
        style.configure('TLabelframe.Label', background=BG_DARK, foreground=FG_ACCENT, font=('Segoe UI', 10, 'bold'))

        style.configure('TLabel', background=BG_DARK, foreground=FG_PRIMARY)

        # Primary button (green)
        style.configure('TButton', background=BG_BUTTON, foreground=FG_PRIMARY, padding=(12, 6),
                        borderwidth=0, font=('Segoe UI', 9, 'bold'))
        style.map('TButton',
                  background=[('active', '#2ea043'), ('pressed', '#238636'), ('disabled', BG_INPUT)],
                  foreground=[('disabled', FG_SECONDARY)])

        # Secondary button style
        style.configure('Secondary.TButton', background=BG_BUTTON_SEC, foreground=FG_PRIMARY,
                        padding=(12, 6), borderwidth=1, font=('Segoe UI', 9))
        style.map('Secondary.TButton',
                  background=[('active', '#30363d'), ('pressed', '#21262d')])

        # Input fields
        style.configure('TEntry', fieldbackground=BG_INPUT, foreground=FG_PRIMARY,
                        insertcolor=FG_PRIMARY, bordercolor=BORDER, padding=5)
        style.configure('TSpinbox', fieldbackground=BG_INPUT, foreground=FG_PRIMARY,
                        bordercolor=BORDER, arrowcolor=FG_SECONDARY)
        style.configure('TCombobox', fieldbackground=BG_INPUT, foreground=FG_PRIMARY,
                        background=BG_INPUT, bordercolor=BORDER, arrowcolor=FG_SECONDARY)
        style.map('TCombobox', fieldbackground=[('readonly', BG_INPUT)])

        # Checkbuttons
        style.configure('TCheckbutton', background=BG_DARK, foreground=FG_PRIMARY)
        style.map('TCheckbutton', background=[('active', BG_DARK)])

        # Custom label styles
        style.configure('Header.TLabel', font=('Segoe UI', 11, 'bold'), background=BG_DARK, foreground=FG_PRIMARY)
        style.configure('Status.TLabel', font=('Segoe UI', 10), background=BG_DARK, foreground=FG_SECONDARY)
        style.configure('Credit.TLabel', font=('Segoe UI', 11, 'bold'), background=BG_DARK, foreground=SUCCESS)
        style.configure('Big.TLabel', font=('Segoe UI', 20, 'bold'), background=BG_DARK, foreground=FG_PRIMARY)
        style.configure('Metric.TLabel', font=('Segoe UI', 11), background=BG_DARK, foreground=FG_SECONDARY)
        style.configure('Running.TLabel', foreground=SUCCESS, background=BG_DARK, font=('Segoe UI', 10, 'bold'))
        style.configure('Stopped.TLabel', foreground=ERROR, background=BG_DARK, font=('Segoe UI', 10, 'bold'))
        style.configure('Link.TLabel', foreground=FG_ACCENT, background=BG_DARK, font=('Segoe UI', 10, 'underline'))

        # Store colors for dialogs
        self.colors = {
            'bg': BG_DARK, 'card': BG_CARD, 'input': BG_INPUT,
            'fg': FG_PRIMARY, 'fg2': FG_SECONDARY, 'accent': FG_ACCENT,
            'success': SUCCESS, 'error': ERROR, 'border': BORDER
        }

        # Menu bar
        menubar = tk.Menu(self.root, bg=BG_CARD, fg=FG_PRIMARY,
                          activebackground=FG_ACCENT, activeforeground=BG_DARK,
                          borderwidth=0, relief=tk.FLAT)
        self.root.config(menu=menubar)

        menu_style = {'tearoff': 0, 'bg': BG_CARD, 'fg': FG_PRIMARY,
                      'activebackground': FG_ACCENT, 'activeforeground': BG_DARK,
                      'borderwidth': 0, 'relief': tk.FLAT}

        file_menu = tk.Menu(menubar, **menu_style)
        menubar.add_cascade(label="File", menu=file_menu)
        file_menu.add_command(label="Exit", command=self._on_close)

        self.account_menu = tk.Menu(menubar, **menu_style)
        self.menu_style = menu_style  # Save for rebuilding menu
        menubar.add_cascade(label="Account", menu=self.account_menu)
        self._update_account_menu()

        settings_menu = tk.Menu(menubar, **menu_style)
        menubar.add_cascade(label="Settings", menu=settings_menu)
        settings_menu.add_command(label="Training Settings...", command=self.show_settings)
        settings_menu.add_separator()

        # Theme submenu
        theme_menu = tk.Menu(settings_menu, **menu_style)
        settings_menu.add_cascade(label="Theme", menu=theme_menu)
        self.theme_var = tk.StringVar(value=self.config.streaming.theme)
        theme_menu.add_radiobutton(label="Dark", variable=self.theme_var, value="dark", command=self._apply_theme)
        theme_menu.add_radiobutton(label="Light", variable=self.theme_var, value="light", command=self._apply_theme)

        help_menu = tk.Menu(menubar, **menu_style)
        menubar.add_cascade(label="Help", menu=help_menu)
        help_menu.add_command(label="Check for Updates...", command=self.check_for_updates)
        help_menu.add_separator()
        help_menu.add_command(label="About", command=self.show_about)

        # Main container
        main_frame = ttk.Frame(self.root, padding="10")
        main_frame.pack(fill=tk.BOTH, expand=True)

        # === Top Section: Status and Controls ===
        top_frame = ttk.Frame(main_frame)
        top_frame.pack(fill=tk.X, pady=(0, 10))

        # Status panel (left)
        status_frame = ttk.LabelFrame(top_frame, text="Account", padding="10")
        status_frame.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(0, 10))

        self.status_label = ttk.Label(status_frame, text="Not connected", style='Status.TLabel')
        self.status_label.pack(anchor=tk.W, pady=2)

        self.user_label = ttk.Label(status_frame, text="", style='Status.TLabel')
        self.user_label.pack(anchor=tk.W, pady=2)

        self.credit_label = ttk.Label(status_frame, text="", style='Credit.TLabel')
        self.credit_label.pack(anchor=tk.W, pady=2)

        # Clickable link to website
        self.link_label = ttk.Label(status_frame, text="axiom.heliex.net", style='Link.TLabel', cursor="hand2")
        self.link_label.pack(anchor=tk.W, pady=(5, 2))
        self.link_label.bind("<Button-1>", lambda e: self._open_website())

        # Account buttons frame
        self.account_btn_frame = ttk.Frame(status_frame)
        self.account_btn_frame.pack(anchor=tk.W, pady=(5, 0))

        # Login/Register buttons (shown when logged out)
        self.login_btn = ttk.Button(self.account_btn_frame, text="Login", command=self.show_login, width=8)
        self.register_btn = ttk.Button(self.account_btn_frame, text="Register", command=self.show_register, width=8)

        # Logout button (shown when logged in)
        self.logout_btn = ttk.Button(self.account_btn_frame, text="Logout", command=self.logout, width=8)

        # Training controls (middle)
        controls_frame = ttk.LabelFrame(top_frame, text="Training Controls", padding="10")
        controls_frame.pack(side=tk.LEFT, fill=tk.Y, padx=(0, 10))

        # Workers control with +/- buttons
        workers_ctrl_frame = ttk.Frame(controls_frame)
        workers_ctrl_frame.pack(fill=tk.X, pady=(0, 10))

        ttk.Label(workers_ctrl_frame, text="Workers:", style='Status.TLabel').grid(row=0, column=0, sticky=tk.W)

        self.max_workers = os.cpu_count() or 8
        self.workers_var = tk.StringVar(value=str(self.config.streaming.num_workers))

        workers_input_frame = ttk.Frame(workers_ctrl_frame)
        workers_input_frame.grid(row=0, column=1, padx=(10, 0))

        ttk.Button(workers_input_frame, text="-", width=2, command=self._decrement_workers, style='Secondary.TButton').pack(side=tk.LEFT)
        self.workers_entry = ttk.Entry(workers_input_frame, textvariable=self.workers_var, width=4,
                                        font=('Segoe UI', 10), justify=tk.CENTER)
        self.workers_entry.pack(side=tk.LEFT, padx=3)
        ttk.Button(workers_input_frame, text="+", width=2, command=self._increment_workers, style='Secondary.TButton').pack(side=tk.LEFT)
        ttk.Label(workers_input_frame, text=f"/{self.max_workers}", style='Status.TLabel').pack(side=tk.LEFT, padx=(5, 0))
        ttk.Button(workers_input_frame, text="Apply", command=self._apply_workers, width=6).pack(side=tk.LEFT, padx=(10, 0))

        # Start/Stop buttons
        btn_frame = ttk.Frame(controls_frame)
        btn_frame.pack(fill=tk.X)

        self.start_btn = ttk.Button(btn_frame, text="Start Training", command=self.start_training)
        self.start_btn.pack(side=tk.LEFT, padx=(0, 5), fill=tk.X, expand=True)

        self.suspend_btn = ttk.Button(btn_frame, text="Suspend", command=self.toggle_suspend, style='Secondary.TButton', state=tk.DISABLED)
        self.suspend_btn.pack(side=tk.LEFT, padx=(0, 5), fill=tk.X, expand=True)

        self.stop_btn = ttk.Button(btn_frame, text="Stop", command=self.stop_training, style='Secondary.TButton', state=tk.DISABLED)
        self.stop_btn.pack(side=tk.LEFT, fill=tk.X, expand=True)

        # Resource Settings (right)
        endo_frame = ttk.LabelFrame(top_frame, text="Resource Settings", padding="10")
        endo_frame.pack(side=tk.RIGHT, fill=tk.Y)

        # Use grid for alignment
        ttk.Label(endo_frame, text="Max Workers %:", style='Status.TLabel').grid(row=0, column=0, sticky=tk.W, pady=2)
        self.ram_threshold_var = tk.StringVar(value=str(int(self.config.streaming.memory_threshold_pct)))
        ram_entry = ttk.Entry(endo_frame, textvariable=self.ram_threshold_var, width=4, justify=tk.CENTER)
        ram_entry.grid(row=0, column=1, padx=5, pady=2)
        ttk.Label(endo_frame, text="%", style='Status.TLabel').grid(row=0, column=2, sticky=tk.W, pady=2)

        # CPU Mode display
        ttk.Label(endo_frame, text="CPU Mode:", style='Status.TLabel').grid(row=1, column=0, sticky=tk.W, pady=2)
        cpu_mode_text = f"{self.config.streaming.cpu_mode.title()}"
        if self.config.streaming.cpu_mode == "fixed":
            cpu_mode_text += f" ({int(self.config.streaming.cpu_fixed_pct)}%)"
        else:
            cpu_mode_text += f" (>{int(self.config.streaming.cpu_backoff_threshold)}%)"
        self.cpu_mode_label = ttk.Label(endo_frame, text=cpu_mode_text, style='Status.TLabel')
        self.cpu_mode_label.grid(row=1, column=1, columnspan=2, sticky=tk.W, padx=5, pady=2)

        # Save button
        ttk.Button(endo_frame, text="Save", command=self._save_endo_settings).grid(row=2, column=0, columnspan=3, pady=(10, 5))

        # Training status indicator
        status_frame = ttk.Frame(endo_frame)
        status_frame.grid(row=3, column=0, columnspan=3, pady=(5, 0))
        ttk.Label(status_frame, text="Status:", style='Status.TLabel').pack(side=tk.LEFT)
        self.training_status_label = ttk.Label(status_frame, text="Stopped", style='Stopped.TLabel')
        self.training_status_label.pack(side=tk.LEFT, padx=(5, 0))

        # === Update Banner (hidden by default) ===
        self.update_frame = ttk.Frame(main_frame)
        # Don't pack yet - will show when update is available

        self.update_banner = ttk.Label(
            self.update_frame,
            text="",
            font=('Segoe UI', 10),
            foreground='#4fc3f7',
            background='#0f3460',
            padding=(10, 5)
        )
        self.update_banner.pack(side=tk.LEFT, fill=tk.X, expand=True)

        self.update_btn = ttk.Button(self.update_frame, text="View Update", command=self.show_update_dialog, width=12)
        self.update_btn.pack(side=tk.RIGHT, padx=(10, 0))

        # === Metrics Dashboard ===
        metrics_frame = ttk.LabelFrame(main_frame, text="Training Metrics", padding="15")
        metrics_frame.pack(fill=tk.X, pady=(0, 10))

        # Grid of metrics
        metrics_grid = ttk.Frame(metrics_frame)
        metrics_grid.pack(fill=tk.X)

        # Row 1: Samples, Syncs, Rate
        ttk.Label(metrics_grid, text="Samples Processed", style='Status.TLabel').grid(row=0, column=0, padx=20, sticky=tk.W)
        ttk.Label(metrics_grid, text="Weight Syncs", style='Status.TLabel').grid(row=0, column=1, padx=20, sticky=tk.W)
        ttk.Label(metrics_grid, text="Model Architecture", style='Status.TLabel').grid(row=0, column=2, padx=20, sticky=tk.W)

        self.samples_label = ttk.Label(metrics_grid, text="0", style='Big.TLabel')
        self.samples_label.grid(row=1, column=0, padx=20, sticky=tk.W)

        self.syncs_label = ttk.Label(metrics_grid, text="0", style='Big.TLabel')
        self.syncs_label.grid(row=1, column=1, padx=20, sticky=tk.W)

        self.arch_label = ttk.Label(metrics_grid, text="Transformer 42.6M", style='Metric.TLabel')
        self.arch_label.grid(row=1, column=2, padx=20, sticky=tk.W)

        # Row 2: CPU, Memory
        ttk.Label(metrics_grid, text="CPU Usage", style='Status.TLabel').grid(row=2, column=0, padx=20, pady=(15, 0), sticky=tk.W)
        ttk.Label(metrics_grid, text="Memory Usage", style='Status.TLabel').grid(row=2, column=1, padx=20, pady=(15, 0), sticky=tk.W)
        ttk.Label(metrics_grid, text="Server", style='Status.TLabel').grid(row=2, column=2, padx=20, pady=(15, 0), sticky=tk.W)

        self.cpu_label = ttk.Label(metrics_grid, text="0%", style='Big.TLabel')
        self.cpu_label.grid(row=3, column=0, padx=20, sticky=tk.W)

        self.memory_label = ttk.Label(metrics_grid, text="0%", style='Big.TLabel')
        self.memory_label.grid(row=3, column=1, padx=20, sticky=tk.W)

        self.server_label = ttk.Label(metrics_grid, text="65.21.196.61:8765", style='Metric.TLabel')
        self.server_label.grid(row=3, column=2, padx=20, sticky=tk.W)

        # Row 4: Disk Space
        ttk.Label(metrics_grid, text="Disk Space", style='Status.TLabel').grid(row=4, column=0, padx=20, pady=(15, 0), sticky=tk.W)
        ttk.Label(metrics_grid, text="", style='Status.TLabel').grid(row=4, column=1, padx=20, pady=(15, 0), sticky=tk.W)
        ttk.Label(metrics_grid, text="", style='Status.TLabel').grid(row=4, column=2, padx=20, pady=(15, 0), sticky=tk.W)

        self.disk_label = ttk.Label(metrics_grid, text="--% free", style='Big.TLabel')
        self.disk_label.grid(row=5, column=0, padx=20, sticky=tk.W)

        self.disk_note_label = ttk.Label(metrics_grid, text="Cache auto-clears on stop", style='Status.TLabel', foreground='gray')
        self.disk_note_label.grid(row=5, column=1, columnspan=2, padx=20, sticky=tk.W)

        # === Contribute Notice ===
        contribute_frame = ttk.Frame(main_frame)
        contribute_frame.pack(fill=tk.X, pady=(0, 5))

        ttk.Label(contribute_frame, text="Contribute data for training by adding files to:",
                  style='Status.TLabel').pack(side=tk.LEFT)

        self.contribute_link = ttk.Label(contribute_frame, text=str(CONTRIBUTE_DIR),
                                          style='Link.TLabel', cursor="hand2")
        self.contribute_link.pack(side=tk.LEFT, padx=(5, 0))
        self.contribute_link.bind("<Button-1>", lambda e: self._open_contribute_folder())

        # Add to copyable labels
        self.contribute_link.bind("<Button-3>", self._show_label_context_menu)

        # === Log Section ===
        log_frame = ttk.LabelFrame(main_frame, text="Activity Log", padding="5")
        log_frame.pack(fill=tk.BOTH, expand=True)

        # Log controls
        log_controls = ttk.Frame(log_frame)
        log_controls.pack(fill=tk.X, pady=(0, 5))

        ttk.Button(log_controls, text="Copy Log", command=self._copy_log, width=10).pack(side=tk.LEFT)
        ttk.Button(log_controls, text="Clear Log", command=self._clear_log, width=10).pack(side=tk.RIGHT)

        # Log text
        self.log_text = tk.Text(log_frame, wrap=tk.WORD, font=('Consolas', 9),
                                bg='#0d1117', fg='#f0f6fc', relief=tk.SUNKEN, bd=1,
                                insertbackground='white')
        log_scrollbar = ttk.Scrollbar(log_frame, orient=tk.VERTICAL, command=self.log_text.yview)
        self.log_text.configure(yscrollcommand=log_scrollbar.set)
        self.log_text.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        log_scrollbar.pack(side=tk.RIGHT, fill=tk.Y)

        # Configure log colors
        self.log_text.tag_configure('error', foreground='#f85149')
        self.log_text.tag_configure('success', foreground='#3fb950')
        self.log_text.tag_configure('info', foreground='#58a6ff')
        self.log_text.tag_configure('warning', foreground='#d29922')
        self.log_text.tag_configure('default', foreground='#8b949e')

        # Right-click context menu for log
        self.log_context_menu = tk.Menu(self.log_text, tearoff=0, bg=BG_CARD, fg=FG_PRIMARY,
                                         activebackground=FG_ACCENT, activeforeground=BG_DARK)
        self.log_context_menu.add_command(label="Copy", command=self._copy_selection, accelerator="Ctrl+C")
        self.log_context_menu.add_command(label="Select All", command=self._select_all_log, accelerator="Ctrl+A")
        self.log_context_menu.add_separator()
        self.log_context_menu.add_command(label="Copy All", command=self._copy_log)

        self.log_text.bind("<Button-3>", self._show_log_context_menu)
        self.log_text.bind("<Control-c>", lambda e: self._copy_selection())
        self.log_text.bind("<Control-a>", lambda e: self._select_all_log())

        # Make labels copyable with right-click
        self._label_context_menu = tk.Menu(self.root, tearoff=0, bg=BG_CARD, fg=FG_PRIMARY,
                                            activebackground=FG_ACCENT, activeforeground=BG_DARK)
        self._label_context_menu.add_command(label="Copy", command=self._copy_label_text)
        self._current_label = None

        # Add copy menu to all important labels
        copyable_labels = [
            self.status_label, self.user_label, self.credit_label,
            self.samples_label, self.syncs_label, self.arch_label,
            self.cpu_label, self.memory_label, self.server_label,
            self.disk_label, self.training_status_label
        ]
        for label in copyable_labels:
            label.bind("<Button-3>", self._show_label_context_menu)

    def _show_label_context_menu(self, event):
        """Show right-click context menu for labels"""
        self._current_label = event.widget
        try:
            self._label_context_menu.tk_popup(event.x_root, event.y_root)
        finally:
            self._label_context_menu.grab_release()

    def _copy_label_text(self):
        """Copy label text to clipboard"""
        if self._current_label:
            text = self._current_label.cget("text")
            if text:
                self.root.clipboard_clear()
                self.root.clipboard_append(text)

    def _show_log_context_menu(self, event):
        """Show right-click context menu"""
        try:
            self.log_context_menu.tk_popup(event.x_root, event.y_root)
        finally:
            self.log_context_menu.grab_release()

    def _copy_selection(self):
        """Copy selected text to clipboard"""
        try:
            selected = self.log_text.get(tk.SEL_FIRST, tk.SEL_LAST)
            if selected:
                self.root.clipboard_clear()
                self.root.clipboard_append(selected)
        except tk.TclError:
            pass  # No selection

    def _select_all_log(self):
        """Select all text in log"""
        self.log_text.tag_add(tk.SEL, "1.0", tk.END)
        self.log_text.mark_set(tk.INSERT, "1.0")
        self.log_text.see(tk.INSERT)
        return "break"

    def _apply_theme(self):
        """Apply the selected theme"""
        theme = self.theme_var.get()
        self.config.streaming.theme = theme
        self.config.save()

        # Define color palettes
        if theme == "light":
            # Clean system-like light theme (matches original Axiom.exe)
            colors = {
                'bg': '#f0f0f0',          # System gray background
                'card': '#ffffff',         # White cards
                'input': '#ffffff',        # White inputs
                'fg': '#000000',           # Black text
                'fg2': '#444444',          # Dark gray secondary text
                'accent': '#0078d4',       # Windows blue
                'success': '#2e7d32',      # Green (same as original)
                'error': '#c62828',        # Red
                'warning': '#ef6c00',      # Orange
                'border': '#cccccc',       # Light gray border
                'button': '#0078d4',       # Windows blue button
                'button_hover': '#106ebe', # Darker blue hover
                'log_bg': '#fafafa',       # Light gray log background
            }
        else:  # dark
            colors = {
                'bg': '#0d1117',
                'card': '#161b22',
                'input': '#21262d',
                'fg': '#f0f6fc',
                'fg2': '#8b949e',
                'accent': '#58a6ff',
                'success': '#3fb950',
                'error': '#f85149',
                'warning': '#d29922',
                'border': '#30363d',
                'button': '#238636',
                'button_hover': '#2ea043',
                'log_bg': '#0d1117',
            }

        self.colors = colors

        # Update root background
        self.root.configure(bg=colors['bg'])

        # Update ttk styles
        style = ttk.Style()

        style.configure('.', background=colors['bg'], foreground=colors['fg'],
                        fieldbackground=colors['input'], bordercolor=colors['border'])
        style.configure('TFrame', background=colors['bg'])
        style.configure('TLabelframe', background=colors['bg'], foreground=colors['fg'], bordercolor=colors['border'])
        style.configure('TLabelframe.Label', background=colors['bg'], foreground=colors['accent'], font=('Segoe UI', 10, 'bold'))
        style.configure('TLabel', background=colors['bg'], foreground=colors['fg'])
        style.configure('TButton', background=colors['button'], foreground='#ffffff' if theme == 'dark' else '#ffffff',
                        padding=(12, 6), borderwidth=0, font=('Segoe UI', 9, 'bold'))
        style.map('TButton',
                  background=[('active', colors['button_hover']), ('pressed', colors['button']), ('disabled', colors['input'])],
                  foreground=[('disabled', colors['fg2'])])
        style.configure('Secondary.TButton', background=colors['input'], foreground=colors['fg'],
                        padding=(12, 6), borderwidth=1, font=('Segoe UI', 9))
        style.map('Secondary.TButton',
                  background=[('active', colors['border']), ('pressed', colors['input'])])
        style.configure('TEntry', fieldbackground=colors['input'], foreground=colors['fg'],
                        insertcolor=colors['fg'], bordercolor=colors['border'], padding=5)
        style.configure('TSpinbox', fieldbackground=colors['input'], foreground=colors['fg'],
                        bordercolor=colors['border'], arrowcolor=colors['fg2'])
        style.configure('TCombobox', fieldbackground=colors['input'], foreground=colors['fg'],
                        background=colors['input'], bordercolor=colors['border'], arrowcolor=colors['fg2'])
        style.map('TCombobox', fieldbackground=[('readonly', colors['input'])])
        style.configure('TCheckbutton', background=colors['bg'], foreground=colors['fg'])
        style.map('TCheckbutton', background=[('active', colors['bg'])])

        # Custom label styles
        style.configure('Header.TLabel', font=('Segoe UI', 11, 'bold'), background=colors['bg'], foreground=colors['fg'])
        style.configure('Status.TLabel', font=('Segoe UI', 10), background=colors['bg'], foreground=colors['fg2'])
        style.configure('Credit.TLabel', font=('Segoe UI', 11, 'bold'), background=colors['bg'], foreground=colors['success'])
        style.configure('Big.TLabel', font=('Segoe UI', 20, 'bold'), background=colors['bg'], foreground=colors['fg'])
        style.configure('Metric.TLabel', font=('Segoe UI', 11), background=colors['bg'], foreground=colors['fg2'])
        style.configure('Running.TLabel', foreground=colors['success'], background=colors['bg'], font=('Segoe UI', 10, 'bold'))
        style.configure('Stopped.TLabel', foreground=colors['error'], background=colors['bg'], font=('Segoe UI', 10, 'bold'))
        style.configure('Link.TLabel', foreground=colors['accent'], background=colors['bg'], font=('Segoe UI', 10, 'underline'))

        # Update log text widget colors
        if hasattr(self, 'log_text'):
            self.log_text.configure(bg=colors['log_bg'], fg=colors['fg'])
            self.log_text.tag_configure('error', foreground=colors['error'])
            self.log_text.tag_configure('success', foreground=colors['success'])
            self.log_text.tag_configure('info', foreground=colors['accent'])
            self.log_text.tag_configure('warning', foreground=colors['warning'])
            self.log_text.tag_configure('default', foreground=colors['fg2'])

        self.log(f"Theme changed to {theme}", 'info')

    def log(self, message, level='default'):
        timestamp = time.strftime('%H:%M:%S')
        line = f"[{timestamp}] {message}\n"

        self.log_text.insert(tk.END, line, level)

        # Trim old lines
        line_count = int(self.log_text.index('end-1c').split('.')[0])
        if line_count > 500:
            self.log_text.delete('1.0', f'{line_count - 500}.0')

        self.log_text.see(tk.END)

    def _thread_safe_log(self, message, level='default'):
        self.root.after(0, lambda m=message, l=level: self.log(m, l))

    def _schedule_refresh(self):
        self.root.after(0, self._refresh_display)

    def _open_website(self):
        """Open the Axiom website in browser"""
        import webbrowser
        webbrowser.open("https://axiom.heliex.net")
        self.log("Opening axiom.heliex.net in browser", 'info')

    def _open_contribute_folder(self):
        """Open the contribute folder in Explorer"""
        CONTRIBUTE_DIR.mkdir(parents=True, exist_ok=True)
        os.startfile(str(CONTRIBUTE_DIR))
        self.log(f"Opening contribute folder: {CONTRIBUTE_DIR}", 'info')

    def _increment_workers(self):
        """Increment worker count"""
        try:
            current = int(self.workers_var.get())
        except ValueError:
            current = 1
        if current < self.max_workers:
            self.workers_var.set(str(current + 1))

    def _decrement_workers(self):
        """Decrement worker count"""
        try:
            current = int(self.workers_var.get())
        except ValueError:
            current = 2
        if current > 1:
            self.workers_var.set(str(current - 1))

    def _apply_workers(self):
        """Apply the worker count setting"""
        try:
            input_count = int(self.workers_var.get())
        except ValueError:
            input_count = self.config.streaming.num_workers
        new_count = max(1, min(input_count, self.max_workers))
        self.workers_var.set(str(new_count))
        self.config.streaming.num_workers = new_count
        self.config.save()
        if input_count > self.max_workers:
            self.log(f"Worker count clamped to max threads ({self.max_workers})", 'warning')
        else:
            self.log(f"Worker count set to {new_count}", 'success')

        # If training is running, need to restart
        if self.streaming_manager.running:
            self.log("Restart training to apply new worker count", 'warning')

    def _save_endo_settings(self):
        """Save resource (RAM threshold) settings"""
        try:
            ram_threshold = int(self.ram_threshold_var.get())
            ram_threshold = max(50, min(99, ram_threshold))
            self.ram_threshold_var.set(str(ram_threshold))
            self.config.streaming.memory_threshold_pct = float(ram_threshold)

            self.config.save()
            self.log(f"Resource settings saved: RAM GC={ram_threshold}%", 'success')

            if self.streaming_manager.running:
                self.log("Restart training to apply new thresholds", 'warning')
        except ValueError:
            self.log("Invalid threshold values - please enter numbers", 'error')

    def _update_cpu_mode_label(self):
        """Update the CPU mode label in the Resource Settings card"""
        cpu_mode_text = f"{self.config.streaming.cpu_mode.title()}"
        if self.config.streaming.cpu_mode == "fixed":
            cpu_mode_text += f" ({int(self.config.streaming.cpu_fixed_pct)}%)"
        else:
            cpu_mode_text += f" (>{int(self.config.streaming.cpu_backoff_threshold)}%)"
        self.cpu_mode_label.config(text=cpu_mode_text)

    def _refresh_display(self):
        """Update the display with current stats"""
        sm = self.streaming_manager

        # Update training status
        if sm.running:
            if self.training_suspended:
                self.training_status_label.config(text="Suspended", style='Stopped.TLabel')
                self.suspend_btn.config(text="Resume")
            else:
                self.training_status_label.config(text=sm.get_status_summary(), style='Running.TLabel')
                self.suspend_btn.config(text="Suspend")
            self.start_btn.config(state=tk.DISABLED)
            self.stop_btn.config(state=tk.NORMAL)
            self.suspend_btn.config(state=tk.NORMAL)
        else:
            self.training_status_label.config(text="Stopped", style='Stopped.TLabel')
            self.start_btn.config(state=tk.NORMAL)
            self.stop_btn.config(state=tk.DISABLED)
            self.suspend_btn.config(state=tk.DISABLED)
            self.training_suspended = False

        # Update workers display
        if hasattr(self, 'workers_var'):
            active_workers = len([w for w in sm.workers.values() if w.connected])
            # Only update if not manually changed
            if not sm.running:
                pass  # Keep user's setting when stopped

        # Update metrics
        self.samples_label.config(text=f"{sm.total_samples:,}")
        self.syncs_label.config(text=f"{sm.total_syncs:,}")
        self.cpu_label.config(text=f"{sm.cpu_percent:.0f}%")
        self.memory_label.config(text=f"{sm.memory_percent:.0f}% ({sm.memory_gb:.1f}GB)")

        # Update disk space display
        self._update_disk_display()

        # Update architecture info
        params_m = 42.6 if self.config.streaming.expert_type == "transformer" else 2.4
        self.arch_label.config(text=f"{self.config.streaming.expert_type.title()} {params_m}M")

    def _update_account_menu(self):
        """Update account menu based on login state"""
        self.account_menu.delete(0, tk.END)
        if self.config.authenticator:
            # Logged in - show logout only
            self.account_menu.add_command(label="Logout", command=self.logout)
        else:
            # Logged out - show login/register
            self.account_menu.add_command(label="Login...", command=self.show_login)
            self.account_menu.add_command(label="Register...", command=self.show_register)
            self.account_menu.add_command(label="Forgot Password...", command=self.show_forgot_password)

    def _fetch_credit_background(self, log_result=False):
        """Fetch credit from server in background thread"""
        try:
            success, total, expavg = self.account_manager.fetch_credit()
            if success:
                # Schedule UI update on main thread
                self.root.after(0, self._update_credit_display)
                if log_result:
                    self.root.after(0, lambda: self.log(f"Credit retrieved: {total:.2f} (avg: {expavg:.2f})", 'success'))
            else:
                if log_result:
                    err = getattr(self.account_manager, 'last_error', 'Unknown error')
                    self.root.after(0, lambda: self.log(f"Credit fetch failed: {err}", 'warning'))
        except Exception as e:
            if log_result:
                self.root.after(0, lambda: self.log(f"Credit fetch error: {e}", 'error'))

    def _update_credit_display(self):
        """Update the credit label with current value"""
        if self.config.authenticator:
            self.credit_label.config(text=f"Credit: {self.config.total_credit:.2f}")

    def _update_status(self):
        """Update account status display and buttons"""
        self._update_account_menu()
        if self.config.authenticator:
            # Logged in
            self.status_label.config(text=f"Connected to {PROJECT_NAME}")
            if self.config.username:
                self.user_label.config(text=f"User: {self.config.username}")
            else:
                self.user_label.config(text=f"Email: {self.config.email}")
            self.credit_label.config(text=f"Credit: {self.config.total_credit:.2f}")
            # Show logout, hide login/register
            self.login_btn.pack_forget()
            self.register_btn.pack_forget()
            self.logout_btn.pack(side=tk.LEFT, padx=2)
        else:
            # Logged out
            self.status_label.config(text="Not logged in")
            self.user_label.config(text="Please login to continue")
            self.credit_label.config(text="")
            # Show login/register, hide logout
            self.logout_btn.pack_forget()
            self.login_btn.pack(side=tk.LEFT, padx=2)
            self.register_btn.pack(side=tk.LEFT, padx=2)

    def start_training(self):
        if not self.config.authenticator:
            messagebox.showwarning("Not Logged In", "Please login first to contribute.")
            self.show_login()
            return

        # Check for compressed files
        if not self.config.streaming.hide_compressed_warning:
            compressed = self._find_compressed_files()
            if compressed:
                if not self._show_compressed_warning(compressed):
                    return  # User cancelled

        # Check disk space
        if not self._check_disk_space():
            return

        self.log("Starting Hebbian training...", 'info')
        self.log(f"  Workers: {self.config.streaming.num_workers}", 'info')
        self.log(f"  Expert Type: {self.config.streaming.expert_type}", 'info')
        self.log(f"  Sync Interval: {self.config.streaming.sync_interval} samples", 'info')
        self.log("  Method: 'Neurons that fire together, wire together'", 'info')

        self.streaming_manager.start()
        self._refresh_display()

        # Save running state
        self.config.streaming.was_running = True
        self.config.save()

        # Start periodic refresh
        self._start_refresh_timer()

    def _start_refresh_timer(self):
        """Periodically refresh the display"""
        if self.streaming_manager.running:
            self._refresh_display()

            # Fetch credit from server every 30 seconds
            now = time.time()
            if now - self.last_credit_fetch >= 30 and self.config.authenticator:
                self.last_credit_fetch = now
                # Fetch in background thread to avoid blocking UI
                threading.Thread(target=self._fetch_credit_background, daemon=True).start()

            # Schedule next refresh
            self.root.after(1000, self._start_refresh_timer)

            # Also schedule disk check in 500ms (checks twice per second)
            self.root.after(500, self._check_disk_and_stop)

    def _check_disk_and_stop(self):
        """Check disk space and stop training if needed (runs every 500ms)"""
        if self.streaming_manager.running:
            self._update_disk_display()
            if not self._check_disk_space(warn=True):
                self.stop_training()

    def stop_training(self):
        self.log("Stopping training...", 'warning')
        self.streaming_manager.stop()
        self._refresh_display()

        # Save stopped state
        self.config.streaming.was_running = False
        self.config.save()

    def toggle_suspend(self):
        """Suspend or resume training"""
        if not self.streaming_manager.running:
            return

        try:
            import psutil
            proc = psutil.Process(self.streaming_manager.process.pid)
            children = proc.children(recursive=True)

            if self.training_suspended:
                # Resume
                for child in children:
                    try:
                        child.resume()
                    except:
                        pass
                proc.resume()
                self.training_suspended = False
                self.log("Training resumed", 'success')
            else:
                # Suspend
                for child in children:
                    try:
                        child.suspend()
                    except:
                        pass
                proc.suspend()
                self.training_suspended = True
                self.log("Training suspended", 'warning')

            self._refresh_display()
        except Exception as e:
            self.log(f"Suspend/resume error: {e}", 'error')

    def _update_disk_display(self):
        """Update the disk space display in metrics"""
        try:
            import shutil
            usage = shutil.disk_usage(CONTRIBUTE_DIR)
            free_gb = usage.free / (1024**3)
            min_gb = self.config.streaming.min_storage_gb

            # Color based on how close to limit
            if min_gb > 0 and free_gb < min_gb:
                color = 'red'
            elif min_gb > 0 and free_gb < min_gb + 2:
                color = 'orange'
            else:
                color = None  # Default

            text = f"{free_gb:.1f} GB free"
            self.disk_label.config(text=text)
            if color:
                self.disk_label.config(foreground=color)
            else:
                self.disk_label.config(foreground='')
        except Exception:
            self.disk_label.config(text="-- GB free")

    def _check_disk_space(self, warn=True) -> bool:
        """Check if disk usage is within limits. Auto-scales workers if needed. Returns True if OK."""
        try:
            import shutil
            usage = shutil.disk_usage(CONTRIBUTE_DIR)
            used_pct = (usage.used / usage.total) * 100
            free_gb = usage.free / (1024**3)

            disk_ok = True
            min_gb = self.config.streaming.min_storage_gb
            max_pct = self.config.streaming.max_storage_pct

            # Check if we're over limits
            if (min_gb > 0 and free_gb < min_gb) or (max_pct > 0 and used_pct >= max_pct):
                disk_ok = False

            if not disk_ok:
                # Scale down workers instead of stopping
                current_workers = self.config.streaming.num_workers
                if current_workers > 1:
                    new_workers = current_workers - 1
                    self.config.streaming.num_workers = new_workers
                    self.workers_var.set(str(new_workers))
                    self.config.save()
                    self.log(f"Disk space low ({free_gb:.1f}GB free) - scaled down to {new_workers} workers", 'warning')
                    # Restart with fewer workers
                    self.streaming_manager.stop()
                    self.root.after(1000, self.streaming_manager.start)
                    return True  # Don't stop completely, just scaled down
                else:
                    # Already at 1 worker, must stop
                    if warn:
                        self.log(f"Disk space critical ({free_gb:.1f}GB free) - stopping training", 'error')
                    return False

            return True
        except Exception as e:
            self.log(f"Could not check disk space: {e}", 'warning')
            return True

    def _find_compressed_files(self):
        """Find compressed files in the contribute folder"""
        compressed_exts = (
            '.zip', '.gz', '.gzip', '.bz2', '.xz', '.7z', '.rar', '.tar',
            '.tgz', '.tbz2', '.txz', '.lz', '.lzma', '.lz4', '.zst', '.zstd'
        )
        compressed = []
        if CONTRIBUTE_DIR.exists():
            for filepath in CONTRIBUTE_DIR.rglob('*'):
                if filepath.is_file() and filepath.suffix.lower() in compressed_exts:
                    compressed.append(filepath)
        return compressed

    def _show_compressed_warning(self, compressed_files):
        """Show warning dialog for compressed files. Returns True to continue, False to cancel."""
        dialog = tk.Toplevel(self.root)
        dialog.title("Compressed Files Detected")
        dialog.transient(self.root)
        dialog.grab_set()
        dialog.resizable(False, False)

        # Apply theme
        colors = self.colors
        dialog.configure(bg=colors['bg'])

        # Size and center
        dialog.geometry("450x280")
        dialog.update_idletasks()
        x = self.root.winfo_x() + (self.root.winfo_width() - 450) // 2
        y = self.root.winfo_y() + (self.root.winfo_height() - 280) // 2
        dialog.geometry(f"+{x}+{y}")

        result = {'continue': False}

        # Main frame
        frame = ttk.Frame(dialog, padding="20")
        frame.pack(fill=tk.BOTH, expand=True)

        # Header
        ttk.Label(frame, text="Compressed Files Detected",
                  style='Header.TLabel').pack(anchor=tk.W, pady=(0, 10))

        # Message
        ttk.Label(frame, text="The following files cannot be used for training.\nExtract them to use their contents:",
                  style='Status.TLabel', justify=tk.LEFT).pack(anchor=tk.W, pady=(0, 10))

        # File list (scrollable if many)
        list_frame = ttk.Frame(frame)
        list_frame.pack(fill=tk.BOTH, expand=True, pady=(0, 10))

        file_list = tk.Text(list_frame, height=5, wrap=tk.WORD, font=('Consolas', 9),
                           bg=colors['card'], fg=colors['fg'], relief=tk.SUNKEN, bd=1)
        for f in compressed_files[:10]:  # Show max 10
            file_list.insert(tk.END, f"  - {f.name}\n")
        if len(compressed_files) > 10:
            file_list.insert(tk.END, f"  ... and {len(compressed_files) - 10} more\n")
        file_list.configure(state=tk.DISABLED)
        file_list.pack(fill=tk.BOTH, expand=True)

        # Don't show again checkbox
        dont_show_var = tk.BooleanVar(value=False)
        ttk.Checkbutton(frame, text="Don't show this warning again",
                        variable=dont_show_var).pack(anchor=tk.W, pady=(5, 15))

        # Buttons
        btn_frame = ttk.Frame(frame)
        btn_frame.pack(fill=tk.X)

        def on_continue():
            if dont_show_var.get():
                self.config.streaming.hide_compressed_warning = True
                self.config.save()
            result['continue'] = True
            dialog.destroy()

        def on_cancel():
            dialog.destroy()

        ttk.Button(btn_frame, text="Continue Anyway", command=on_continue).pack(side=tk.RIGHT, padx=(5, 0))
        ttk.Button(btn_frame, text="Cancel", command=on_cancel,
                   style='Secondary.TButton').pack(side=tk.RIGHT)

        # Handle window close
        dialog.protocol("WM_DELETE_WINDOW", on_cancel)

        dialog.wait_window()
        return result['continue']

    def _style_dialog(self, dialog):
        """Apply dark theme to a dialog window"""
        dialog.configure(bg=self.colors['bg'])

    def show_login(self):
        login_win = tk.Toplevel(self.root)
        login_win.title("Login to Axiom")
        login_win.geometry("380x260")
        login_win.transient(self.root)
        login_win.grab_set()
        self._style_dialog(login_win)

        # Center
        login_win.update_idletasks()
        x = (login_win.winfo_screenwidth() - 380) // 2
        y = (login_win.winfo_screenheight() - 260) // 2
        login_win.geometry(f"+{x}+{y}")

        frame = ttk.Frame(login_win, padding="20")
        frame.pack(fill=tk.BOTH, expand=True)

        ttk.Label(frame, text="Login to contribute to Axiom", style='Header.TLabel').grid(row=0, column=0, columnspan=2, pady=(0, 15))

        ttk.Label(frame, text="Email:").grid(row=1, column=0, sticky=tk.W, pady=5)
        email_entry = ttk.Entry(frame, width=30)
        email_entry.grid(row=1, column=1, pady=5)
        if self.config.email:
            email_entry.insert(0, self.config.email)

        ttk.Label(frame, text="Password:").grid(row=2, column=0, sticky=tk.W, pady=5)
        password_entry = ttk.Entry(frame, width=30, show="*")
        password_entry.grid(row=2, column=1, pady=5)

        status_label = ttk.Label(frame, text="", foreground="red")
        status_label.grid(row=3, column=0, columnspan=2, pady=10)

        def do_login():
            email = email_entry.get().strip()
            password = password_entry.get()

            if not email or not password:
                status_label.config(text="Please enter email and password")
                return

            status_label.config(text="Logging in...", foreground="black")
            login_win.update()

            success, message = self.account_manager.login(email, password)

            if success:
                self.log(f"Logged in as {email}", 'success')
                login_win.destroy()
                self._update_status()
                # Fetch and log credit
                self.log("Fetching credit from server...", 'info')
                threading.Thread(target=lambda: self._fetch_credit_background(log_result=True), daemon=True).start()
            else:
                status_label.config(text=message, foreground="red")

        btn_frame = ttk.Frame(frame)
        btn_frame.grid(row=4, column=0, columnspan=2, pady=10)
        ttk.Button(btn_frame, text="Login", command=do_login).pack(side=tk.LEFT, padx=5)
        ttk.Button(btn_frame, text="Register", command=lambda: [login_win.destroy(), self.show_register()]).pack(side=tk.LEFT, padx=5)

        password_entry.bind('<Return>', lambda e: do_login())
        email_entry.focus()

    def show_register(self):
        reg_win = tk.Toplevel(self.root)
        reg_win.title("Register")
        reg_win.geometry("380x340")
        reg_win.transient(self.root)
        reg_win.grab_set()
        self._style_dialog(reg_win)

        reg_win.update_idletasks()
        x = (reg_win.winfo_screenwidth() - 380) // 2
        y = (reg_win.winfo_screenheight() - 340) // 2
        reg_win.geometry(f"+{x}+{y}")

        frame = ttk.Frame(reg_win, padding="20")
        frame.pack(fill=tk.BOTH, expand=True)

        ttk.Label(frame, text="Create an Axiom Account", style='Header.TLabel').grid(row=0, column=0, columnspan=2, pady=(0, 15))

        ttk.Label(frame, text="Email:").grid(row=1, column=0, sticky=tk.W, pady=5)
        email_entry = ttk.Entry(frame, width=30)
        email_entry.grid(row=1, column=1, pady=5)

        ttk.Label(frame, text="Username:").grid(row=2, column=0, sticky=tk.W, pady=5)
        username_entry = ttk.Entry(frame, width=30)
        username_entry.grid(row=2, column=1, pady=5)

        ttk.Label(frame, text="Password:").grid(row=3, column=0, sticky=tk.W, pady=5)
        password_entry = ttk.Entry(frame, width=30, show="*")
        password_entry.grid(row=3, column=1, pady=5)

        ttk.Label(frame, text="Confirm:").grid(row=4, column=0, sticky=tk.W, pady=5)
        confirm_entry = ttk.Entry(frame, width=30, show="*")
        confirm_entry.grid(row=4, column=1, pady=5)

        status_label = ttk.Label(frame, text="", foreground="red")
        status_label.grid(row=5, column=0, columnspan=2, pady=10)

        def do_register():
            email = email_entry.get().strip()
            username = username_entry.get().strip()
            password = password_entry.get()
            confirm = confirm_entry.get()

            if not email or not username or not password:
                status_label.config(text="Please fill in all fields")
                return

            if password != confirm:
                status_label.config(text="Passwords do not match")
                return

            status_label.config(text="Creating account...", foreground="black")
            reg_win.update()

            success, message = self.account_manager.register(email, username, password)

            if success:
                self.log(f"Account created for {email}", 'success')
                reg_win.destroy()
                self._update_status()
            else:
                status_label.config(text=message, foreground="red")

        btn_frame = ttk.Frame(frame)
        btn_frame.grid(row=6, column=0, columnspan=2, pady=10)
        ttk.Button(btn_frame, text="Register", command=do_register).pack(side=tk.LEFT, padx=5)
        ttk.Button(btn_frame, text="Back to Login", command=lambda: [reg_win.destroy(), self.show_login()]).pack(side=tk.LEFT, padx=5)

    def show_forgot_password(self):
        dialog = tk.Toplevel(self.root)
        dialog.title("Forgot Password")
        dialog.geometry("350x150")
        dialog.transient(self.root)
        dialog.grab_set()
        self._style_dialog(dialog)

        dialog.update_idletasks()
        x = (dialog.winfo_screenwidth() - 350) // 2
        y = (dialog.winfo_screenheight() - 150) // 2
        dialog.geometry(f"+{x}+{y}")

        frame = ttk.Frame(dialog, padding="20")
        frame.pack(fill=tk.BOTH, expand=True)

        ttk.Label(frame, text="Email:").grid(row=0, column=0, sticky=tk.W, pady=5)
        email_entry = ttk.Entry(frame, width=30)
        email_entry.grid(row=0, column=1, pady=5)

        status_label = ttk.Label(frame, text="")
        status_label.grid(row=1, column=0, columnspan=2, pady=10)

        def do_reset():
            email = email_entry.get().strip()
            if not email:
                status_label.config(text="Please enter your email", foreground="red")
                return

            success, message = self.account_manager.forgot_password(email)
            status_label.config(text=message, foreground="green" if success else "red")

        ttk.Button(frame, text="Send Reset Email", command=do_reset).grid(row=2, column=0, columnspan=2, pady=10)

    def logout(self):
        if self.streaming_manager.running:
            self.stop_training()
        self.config.logout()
        self._update_status()
        self.log("Logged out", 'info')

    def show_settings(self):
        dialog = tk.Toplevel(self.root)
        dialog.title("Training Settings")
        dialog.geometry("500x650")
        dialog.transient(self.root)
        dialog.grab_set()
        dialog.resizable(True, True)
        dialog.minsize(450, 400)
        self._style_dialog(dialog)

        dialog.update_idletasks()
        x = self.root.winfo_x() + (self.root.winfo_width() - 500) // 2
        y = self.root.winfo_y() + (self.root.winfo_height() - 650) // 2
        dialog.geometry(f"+{x}+{y}")

        # Create scrollable container
        container = ttk.Frame(dialog)
        container.pack(fill=tk.BOTH, expand=True)

        canvas = tk.Canvas(container, highlightthickness=0)
        scrollbar = ttk.Scrollbar(container, orient="vertical", command=canvas.yview)
        frame = ttk.Frame(canvas, padding="20")

        frame.bind("<Configure>", lambda e: canvas.configure(scrollregion=canvas.bbox("all")))
        canvas.create_window((0, 0), window=frame, anchor="nw", width=480)
        canvas.configure(yscrollcommand=scrollbar.set)

        canvas.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)

        # Mouse wheel scrolling
        def on_mousewheel(event):
            canvas.yview_scroll(int(-1 * (event.delta / 120)), "units")
        canvas.bind_all("<MouseWheel>", on_mousewheel)
        dialog.protocol("WM_DELETE_WINDOW", lambda: [canvas.unbind_all("<MouseWheel>"), dialog.destroy()])

        max_cpus = os.cpu_count() or 8

        # === Training Settings Section ===
        training_frame = ttk.LabelFrame(frame, text="Training Settings", padding="10")
        training_frame.pack(fill=tk.X, pady=(0, 10))

        # Workers
        workers_row = ttk.Frame(training_frame)
        workers_row.pack(fill=tk.X, pady=2)
        ttk.Label(workers_row, text="Number of Workers:", width=18, anchor=tk.W).pack(side=tk.LEFT)
        workers_var = tk.StringVar(value=str(self.config.streaming.num_workers))
        ttk.Entry(workers_row, textvariable=workers_var, width=8).pack(side=tk.LEFT, padx=5)
        ttk.Label(workers_row, text=f"(1-{max_cpus})", foreground="gray").pack(side=tk.LEFT)

        # CPU Mode
        cpu_mode_frame = ttk.LabelFrame(training_frame, text="CPU Usage Mode", padding="5")
        cpu_mode_frame.pack(fill=tk.X, pady=(10, 0))

        cpu_mode_var = tk.StringVar(value=self.config.streaming.cpu_mode)

        # Fixed mode row
        fixed_row = ttk.Frame(cpu_mode_frame)
        fixed_row.pack(fill=tk.X, pady=2)
        ttk.Radiobutton(fixed_row, text="Fixed:", variable=cpu_mode_var, value="fixed", width=12).pack(side=tk.LEFT)
        cpu_fixed_var = tk.StringVar(value=str(int(self.config.streaming.cpu_fixed_pct)))
        cpu_fixed_spin = ttk.Spinbox(fixed_row, from_=10, to=100, width=5, textvariable=cpu_fixed_var)
        cpu_fixed_spin.pack(side=tk.LEFT, padx=5)
        ttk.Label(fixed_row, text="% of CPU (always use this amount)").pack(side=tk.LEFT)

        # Adaptive mode row
        adaptive_row = ttk.Frame(cpu_mode_frame)
        adaptive_row.pack(fill=tk.X, pady=2)
        ttk.Radiobutton(adaptive_row, text="Adaptive:", variable=cpu_mode_var, value="adaptive", width=12).pack(side=tk.LEFT)
        ttk.Label(adaptive_row, text="Back off when system CPU >").pack(side=tk.LEFT)
        cpu_backoff_var = tk.StringVar(value=str(int(self.config.streaming.cpu_backoff_threshold)))
        cpu_backoff_spin = ttk.Spinbox(adaptive_row, from_=20, to=95, width=5, textvariable=cpu_backoff_var)
        cpu_backoff_spin.pack(side=tk.LEFT, padx=5)
        ttk.Label(adaptive_row, text="%").pack(side=tk.LEFT)

        ttk.Label(cpu_mode_frame, text="Adaptive yields to other apps; Fixed maintains consistent usage",
                  foreground="gray", font=('Segoe UI', 9)).pack(anchor=tk.W, pady=(5, 0))

        # === Storage Section ===
        storage_frame = ttk.LabelFrame(frame, text="Storage Limit", padding="10")
        storage_frame.pack(fill=tk.X, pady=(0, 10))

        # Min free GB (primary option)
        gb_row = ttk.Frame(storage_frame)
        gb_row.pack(fill=tk.X, pady=2)
        ttk.Label(gb_row, text="Min free space:", width=14, anchor=tk.W).pack(side=tk.LEFT)
        min_storage_var = tk.StringVar(value=str(self.config.streaming.min_storage_gb))
        min_storage_spin = ttk.Spinbox(gb_row, from_=1, to=100, width=5, textvariable=min_storage_var, increment=0.5)
        min_storage_spin.pack(side=tk.LEFT, padx=5)
        ttk.Label(gb_row, text="GB (scale down workers if below this)").pack(side=tk.LEFT)

        # Max percentage (optional)
        pct_row = ttk.Frame(storage_frame)
        pct_row.pack(fill=tk.X, pady=2)
        ttk.Label(pct_row, text="Max disk usage:", width=14, anchor=tk.W).pack(side=tk.LEFT)
        max_storage_var = tk.StringVar(value=str(int(self.config.streaming.max_storage_pct)) if self.config.streaming.max_storage_pct > 0 else "")
        max_storage_entry = ttk.Entry(pct_row, textvariable=max_storage_var, width=5)
        max_storage_entry.pack(side=tk.LEFT, padx=5)
        ttk.Label(pct_row, text="% (optional, scale down if exceeded)").pack(side=tk.LEFT)

        ttk.Label(storage_frame, text="Cache auto-clears when training stops. Checks every 0.5 seconds.",
                  foreground="gray", font=('Segoe UI', 9)).pack(anchor=tk.W, pady=(5, 0))

        # === Schedule Section ===
        schedule_frame = ttk.LabelFrame(frame, text="Schedule", padding="10")
        schedule_frame.pack(fill=tk.X, pady=(0, 10))

        # Enable checkbox
        schedule_enabled_var = tk.BooleanVar(value=self.config.streaming.schedule_enabled)
        ttk.Checkbutton(schedule_frame, text="Enable scheduled training", variable=schedule_enabled_var).pack(anchor=tk.W)

        # Days of week
        days_frame = ttk.Frame(schedule_frame)
        days_frame.pack(fill=tk.X, pady=(10, 5))
        ttk.Label(days_frame, text="Days:", width=8, anchor=tk.W).pack(side=tk.LEFT)

        day_names = ["Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"]
        day_vars = []
        for i, day in enumerate(day_names):
            var = tk.BooleanVar(value=(i in self.config.streaming.schedule_days))
            day_vars.append(var)
            ttk.Checkbutton(days_frame, text=day, variable=var, width=5).pack(side=tk.LEFT)

        # Time range
        time_frame = ttk.Frame(schedule_frame)
        time_frame.pack(fill=tk.X, pady=5)

        ttk.Label(time_frame, text="Start:", width=8, anchor=tk.W).pack(side=tk.LEFT)
        start_hour_var = tk.StringVar(value=f"{self.config.streaming.schedule_start_hour:02d}")
        start_hour_spin = ttk.Spinbox(time_frame, from_=0, to=23, width=3, textvariable=start_hour_var, format="%02.0f")
        start_hour_spin.pack(side=tk.LEFT)
        ttk.Label(time_frame, text=":").pack(side=tk.LEFT)
        start_min_var = tk.StringVar(value=f"{self.config.streaming.schedule_start_minute:02d}")
        start_min_spin = ttk.Spinbox(time_frame, from_=0, to=59, width=3, textvariable=start_min_var, format="%02.0f")
        start_min_spin.pack(side=tk.LEFT)

        ttk.Label(time_frame, text="   End:", width=6, anchor=tk.W).pack(side=tk.LEFT)
        end_hour_var = tk.StringVar(value=f"{self.config.streaming.schedule_end_hour:02d}")
        end_hour_spin = ttk.Spinbox(time_frame, from_=0, to=23, width=3, textvariable=end_hour_var, format="%02.0f")
        end_hour_spin.pack(side=tk.LEFT)
        ttk.Label(time_frame, text=":").pack(side=tk.LEFT)
        end_min_var = tk.StringVar(value=f"{self.config.streaming.schedule_end_minute:02d}")
        end_min_spin = ttk.Spinbox(time_frame, from_=0, to=59, width=3, textvariable=end_min_var, format="%02.0f")
        end_min_spin.pack(side=tk.LEFT)

        # Schedule info
        ttk.Label(schedule_frame, text="Training will auto-start/stop based on this schedule",
                  foreground="gray", font=('Segoe UI', 9)).pack(anchor=tk.W, pady=(5, 0))

        # === Info Section ===
        info_frame = ttk.LabelFrame(frame, text="Current Architecture", padding="10")
        info_frame.pack(fill=tk.X, pady=(0, 10))

        params = "42.6M params/expert" if self.config.streaming.expert_type == "transformer" else "2.4M params/expert"
        total = "17.8B total" if self.config.streaming.expert_type == "transformer" else "1.0B total"
        ttk.Label(info_frame, text=f"420 experts x {params} = {total}").pack(anchor=tk.W)
        ttk.Label(info_frame, text="Hebbian learning: 85% local, 15% gossip sync").pack(anchor=tk.W)

        def save_settings():
            try:
                # Training settings - clamp workers to max threads
                workers_input = int(workers_var.get())
                workers = max(1, min(workers_input, max_cpus))
                if workers_input > max_cpus:
                    workers_var.set(str(workers))  # Update the entry field to show clamped value
                    self.log(f"Worker count clamped to max threads ({max_cpus})", 'warning')
                self.config.streaming.num_workers = workers
                self.workers_var.set(str(workers))  # Update main UI

                # CPU mode settings
                self.config.streaming.cpu_mode = cpu_mode_var.get()
                self.config.streaming.cpu_fixed_pct = float(cpu_fixed_var.get())
                self.config.streaming.cpu_backoff_threshold = float(cpu_backoff_var.get())

                # Storage settings
                self.config.streaming.min_storage_gb = float(min_storage_var.get())
                max_pct_str = max_storage_var.get().strip()
                self.config.streaming.max_storage_pct = float(max_pct_str) if max_pct_str else 0.0

                # Schedule settings
                self.config.streaming.schedule_enabled = schedule_enabled_var.get()

                selected_days = [i for i, var in enumerate(day_vars) if var.get()]
                self.config.streaming.schedule_days = selected_days if selected_days else [0, 1, 2, 3, 4, 5, 6]

                self.config.streaming.schedule_start_hour = int(start_hour_var.get())
                self.config.streaming.schedule_start_minute = int(start_min_var.get())
                self.config.streaming.schedule_end_hour = int(end_hour_var.get())
                self.config.streaming.schedule_end_minute = int(end_min_var.get())

                self.config.save()
                canvas.unbind_all("<MouseWheel>")
                dialog.destroy()
                self.log("Settings saved", 'success')

                # Update CPU mode label in main UI
                self._update_cpu_mode_label()

                if self.config.streaming.schedule_enabled:
                    self._update_schedule_status()
                    self.log(f"Schedule enabled: {self._format_schedule()}", 'info')

            except ValueError:
                messagebox.showerror("Invalid Input", "Please enter valid numbers")

        def cancel_settings():
            canvas.unbind_all("<MouseWheel>")
            dialog.destroy()

        btn_frame = ttk.Frame(frame)
        btn_frame.pack(fill=tk.X, pady=(10, 0))
        ttk.Button(btn_frame, text="Save", command=save_settings).pack(side=tk.RIGHT, padx=5)
        ttk.Button(btn_frame, text="Cancel", command=cancel_settings).pack(side=tk.RIGHT)

    def _format_schedule(self):
        """Format the schedule for display"""
        day_names = ["Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"]
        days = [day_names[i] for i in self.config.streaming.schedule_days]
        start = f"{self.config.streaming.schedule_start_hour:02d}:{self.config.streaming.schedule_start_minute:02d}"
        end = f"{self.config.streaming.schedule_end_hour:02d}:{self.config.streaming.schedule_end_minute:02d}"
        return f"{', '.join(days)} {start}-{end}"

    def _is_scheduled_time(self):
        """Check if current time falls within the schedule"""
        if not self.config.streaming.schedule_enabled:
            return None  # Schedule disabled, no auto control

        now = time.localtime()
        current_day = now.tm_wday  # 0=Monday
        current_minutes = now.tm_hour * 60 + now.tm_min

        start_minutes = self.config.streaming.schedule_start_hour * 60 + self.config.streaming.schedule_start_minute
        end_minutes = self.config.streaming.schedule_end_hour * 60 + self.config.streaming.schedule_end_minute

        # Check if today is a scheduled day
        if current_day not in self.config.streaming.schedule_days:
            return False

        # Check if current time is within range
        if start_minutes <= end_minutes:
            # Normal range (e.g., 09:00 - 17:00)
            return start_minutes <= current_minutes <= end_minutes
        else:
            # Overnight range (e.g., 22:00 - 06:00)
            return current_minutes >= start_minutes or current_minutes <= end_minutes

    def _update_schedule_status(self):
        """Check schedule and start/stop training accordingly"""
        if not self.config.streaming.schedule_enabled:
            return

        should_run = self._is_scheduled_time()

        if should_run and not self.streaming_manager.running:
            self.log("Schedule: Starting training (scheduled time)", 'info')
            self.start_training()
        elif not should_run and self.streaming_manager.running:
            self.log("Schedule: Stopping training (outside scheduled time)", 'info')
            self.stop_training()

    def _schedule_check_loop(self):
        """Periodic check for schedule - runs every minute"""
        if self.config.streaming.schedule_enabled:
            self._update_schedule_status()
        # Check again in 60 seconds
        self.root.after(60000, self._schedule_check_loop)

    def show_about(self):
        dialog = tk.Toplevel(self.root)
        dialog.title("About Axiom")
        dialog.geometry("550x450")
        dialog.transient(self.root)
        dialog.grab_set()
        self._style_dialog(dialog)

        dialog.update_idletasks()
        x = self.root.winfo_x() + (self.root.winfo_width() - 550) // 2
        y = self.root.winfo_y() + (self.root.winfo_height() - 450) // 2
        dialog.geometry(f"+{x}+{y}")

        frame = ttk.Frame(dialog, padding="20")
        frame.pack(fill=tk.BOTH, expand=True)

        ttk.Label(frame, text=f"{APP_NAME}", font=('Segoe UI', 16, 'bold')).pack(pady=(0, 5))
        ttk.Label(frame, text=f"Version {APP_VERSION}").pack(pady=(0, 15))

        text = tk.Text(frame, wrap=tk.WORD, font=('Segoe UI', 10), height=15,
                       bg='#161b22', fg='#f0f6fc', relief=tk.FLAT, padx=10, pady=10,
                       insertbackground='#e8e8e8')
        text.pack(fill=tk.BOTH, expand=True, pady=(0, 15))

        about_text = """HEBBIAN DISTRIBUTED LEARNING

Axiom uses Hebbian learning - "neurons that fire together, wire together" - to train a massive 17.8 billion parameter AI model through distributed computing.

THE MODEL
- 420 specialized experts (like brain regions)
- 42.6 million parameters per expert
- Transformer architecture with self-attention
- Total: 17.8 billion parameters

HOW IT WORKS
Unlike traditional backpropagation, Hebbian learning is local and biologically inspired. Each sample strengthens connections between neurons that activate together.

THE 85/15 SPLIT
- 85% of learning happens locally on your machine
- 15% comes from gossip synchronization with peers
This mimics how biological neural networks learn.

RESOURCE MANAGEMENT
The system monitors CPU and memory usage, automatically managing garbage collection and worker processes based on system load.

YOUR CONTRIBUTION
Every sample you process teaches the model something new. Your CPU performs the computations that power open-source AI research.

https://axiom.heliex.net
"""

        text.insert('1.0', about_text)
        text.configure(state=tk.DISABLED)

        ttk.Button(frame, text="Close", command=dialog.destroy, width=15).pack()

    def _check_updates_background(self):
        """Check for updates in background thread"""
        def check():
            has_update, version, changelog = self.update_manager.check_for_updates()
            if has_update:
                self.root.after(0, lambda: self._show_update_banner(version, changelog))

        threading.Thread(target=check, daemon=True).start()

    def _show_update_banner(self, version, changelog):
        """Show the update notification banner"""
        self.update_banner.config(text=f"Update available: v{version}")
        self.update_frame.pack(fill=tk.X, pady=(0, 10), before=self.root.winfo_children()[0].winfo_children()[1])
        self.log(f"Update available: v{version}", 'info')

    def check_for_updates(self):
        """Manual check for updates from menu"""
        self.log("Checking for updates...", 'info')

        def check():
            has_update, version, changelog = self.update_manager.check_for_updates()
            self.root.after(0, lambda: self._show_update_result(has_update, version, changelog))

        threading.Thread(target=check, daemon=True).start()

    def _show_update_result(self, has_update, version, changelog):
        """Show result of update check"""
        if has_update:
            self._show_update_banner(version, changelog)
            self.show_update_dialog()
        else:
            self.log(f"You're running the latest version (v{APP_VERSION})", 'success')
            messagebox.showinfo("No Updates", f"You're running the latest version.\n\nCurrent: v{APP_VERSION}")

    def show_update_dialog(self):
        """Show dialog with update details and download option"""
        if not self.update_manager.update_available:
            return

        dialog = tk.Toplevel(self.root)
        dialog.title("Update Available")
        dialog.geometry("450x350")
        dialog.transient(self.root)
        dialog.grab_set()
        self._style_dialog(dialog)

        dialog.update_idletasks()
        x = self.root.winfo_x() + (self.root.winfo_width() - 450) // 2
        y = self.root.winfo_y() + (self.root.winfo_height() - 350) // 2
        dialog.geometry(f"+{x}+{y}")

        frame = ttk.Frame(dialog, padding="20")
        frame.pack(fill=tk.BOTH, expand=True)

        # Header
        ttk.Label(frame, text=f"Version {self.update_manager.latest_version} Available",
                  font=('Segoe UI', 14, 'bold')).pack(pady=(0, 5))
        ttk.Label(frame, text=f"Current version: {APP_VERSION}",
                  style='Status.TLabel').pack(pady=(0, 15))

        # Changelog
        ttk.Label(frame, text="What's New:", style='Header.TLabel').pack(anchor=tk.W)

        changelog_text = tk.Text(frame, wrap=tk.WORD, font=('Segoe UI', 10), height=8,
                                  bg='#161b22', fg='#f0f6fc', relief=tk.FLAT, padx=10, pady=10)
        changelog_text.pack(fill=tk.BOTH, expand=True, pady=(5, 15))
        changelog_text.insert('1.0', self.update_manager.changelog or "Bug fixes and improvements.")
        changelog_text.configure(state=tk.DISABLED)

        # Status label
        status_label = ttk.Label(frame, text="", style='Status.TLabel')
        status_label.pack(pady=(0, 10))

        # Buttons
        btn_frame = ttk.Frame(frame)
        btn_frame.pack(fill=tk.X)

        def do_download():
            status_label.config(text="Downloading update...")
            dialog.update()

            # Download to the same directory as this script
            target_dir = Path(__file__).parent

            def download():
                success, message = self.update_manager.download_update(target_dir)
                self.root.after(0, lambda: on_download_complete(success, message))

            threading.Thread(target=download, daemon=True).start()

        def on_download_complete(success, message):
            if success:
                status_label.config(text="Update downloaded! Restart to apply.")
                self.log("Update downloaded successfully", 'success')
                download_btn.config(state=tk.DISABLED)

                # Ask to restart
                if messagebox.askyesno("Restart Now?",
                    "Update downloaded successfully!\n\nRestart now to apply the update?"):
                    dialog.destroy()
                    self._restart_application()
            else:
                status_label.config(text=f"Download failed: {message}")
                self.log(f"Update download failed: {message}", 'error')

        download_btn = ttk.Button(btn_frame, text="Download Update", command=do_download, width=15)
        download_btn.pack(side=tk.LEFT, padx=5)

        ttk.Button(btn_frame, text="Later", command=dialog.destroy, width=10).pack(side=tk.RIGHT, padx=5)

    def _restart_application(self):
        """Restart the application to apply updates"""
        self.log("Restarting application...", 'info')

        # Stop training if running
        if self.streaming_manager.running:
            self.streaming_manager.stop()

        # Launch new instance
        python = sys.executable
        script = Path(__file__)
        subprocess.Popen([python, str(script)], creationflags=subprocess.CREATE_NO_WINDOW)

        # Exit current instance
        self._force_exit()

    def _copy_log(self):
        content = self.log_text.get("1.0", tk.END).strip()
        if content:
            self.root.clipboard_clear()
            self.root.clipboard_append(content)
            self.log("Log copied to clipboard", 'info')

    def _clear_log(self):
        self.log_text.delete("1.0", tk.END)
        self.log("Log cleared", 'info')

    def _on_close(self):
        if self.streaming_manager.running:
            self._show_close_dialog()
        else:
            self._force_exit()

    def _show_close_dialog(self):
        """Show themed close dialog with radio options"""
        dialog = tk.Toplevel(self.root)
        dialog.title("Training Running")
        dialog.transient(self.root)
        dialog.grab_set()
        dialog.resizable(False, False)

        # Apply theme colors
        colors = self.colors
        dialog.configure(bg=colors['bg'])

        # Size and center
        dialog.geometry("350x240")
        dialog.update_idletasks()
        x = self.root.winfo_x() + (self.root.winfo_width() - 350) // 2
        y = self.root.winfo_y() + (self.root.winfo_height() - 240) // 2
        dialog.geometry(f"+{x}+{y}")

        # Main frame
        frame = ttk.Frame(dialog, padding="20")
        frame.pack(fill=tk.BOTH, expand=True)

        # Header
        ttk.Label(frame, text="Training is still running",
                  style='Header.TLabel').pack(anchor=tk.W, pady=(0, 15))

        # Radio options
        choice_var = tk.StringVar(value="stop")

        options = [
            ("stop", "Stop training and exit"),
            ("minimize", "Minimize to taskbar (keep training)"),
            ("cancel", "Don't close"),
        ]

        for value, text in options:
            rb = ttk.Radiobutton(frame, text=text, variable=choice_var, value=value)
            rb.pack(anchor=tk.W, pady=3)

        # Button frame
        btn_frame = ttk.Frame(frame)
        btn_frame.pack(fill=tk.X, pady=(20, 0))

        def on_confirm():
            choice = choice_var.get()
            dialog.destroy()
            if choice == "stop":
                self.stop_training()
                self._force_exit()
            elif choice == "minimize":
                self.root.iconify()
            # cancel = do nothing

        ttk.Button(btn_frame, text="OK", command=on_confirm, width=10).pack(side=tk.RIGHT, padx=(5, 0))
        ttk.Button(btn_frame, text="Cancel", command=dialog.destroy,
                   style='Secondary.TButton', width=10).pack(side=tk.RIGHT)

        # Handle window close button
        dialog.protocol("WM_DELETE_WINDOW", dialog.destroy)

    def _setup_tray(self):
        """Setup system tray icon"""
        # Create a simple icon
        icon_image = self._create_tray_icon()

        # Create menu
        menu = pystray.Menu(
            pystray.MenuItem("Show", self._tray_show, default=True),
            pystray.Menu.SEPARATOR,
            pystray.MenuItem("Start Training", self._tray_start,
                           visible=lambda item: not self.streaming_manager.running),
            pystray.MenuItem("Stop Training", self._tray_stop,
                           visible=lambda item: self.streaming_manager.running),
            pystray.Menu.SEPARATOR,
            pystray.MenuItem("Exit", self._tray_exit)
        )

        self.tray_icon = pystray.Icon(APP_NAME, icon_image, APP_NAME, menu)

        # Run tray in background thread
        tray_thread = threading.Thread(target=self.tray_icon.run, daemon=True)
        tray_thread.start()

    def _create_tray_icon(self):
        """Create a simple tray icon"""
        # Create a 64x64 icon with the Axiom colors
        size = 64
        image = Image.new('RGBA', (size, size), (0, 0, 0, 0))
        draw = ImageDraw.Draw(image)

        # Draw a green circle (Axiom accent color)
        padding = 4
        draw.ellipse([padding, padding, size - padding, size - padding],
                    fill='#3fb950', outline='#238636', width=2)

        # Draw an "A" in the center
        draw.text((size // 2 - 8, size // 2 - 12), "A", fill='white')

        return image

    def _tray_show(self, icon=None, item=None):
        """Show the main window from tray"""
        self.root.after(0, self._show_window)

    def _show_window(self):
        """Show and focus the main window"""
        self.root.deiconify()
        self.root.lift()
        self.root.focus_force()

    def _tray_start(self, icon=None, item=None):
        """Start training from tray"""
        self.root.after(0, self.start_training)

    def _tray_stop(self, icon=None, item=None):
        """Stop training from tray"""
        self.root.after(0, self.stop_training)

    def _tray_exit(self, icon=None, item=None):
        """Exit from tray"""
        if self.tray_icon:
            self.tray_icon.stop()
        self.root.after(0, self._force_exit)

    def _force_exit(self):
        self.streaming_manager.stop()
        if self.tray_icon:
            try:
                self.tray_icon.stop()
            except:
                pass
        try:
            self.root.destroy()
        except:
            pass
        os._exit(0)

    def run(self):
        DATA_DIR.mkdir(parents=True, exist_ok=True)
        CONTRIBUTE_DIR.mkdir(parents=True, exist_ok=True)

        self.root.protocol("WM_DELETE_WINDOW", self._on_close)

        # Log startup
        self.log("Axiom Streaming Client started", 'info')
        self.log(f"Hebbian Learning: 'Neurons that fire together, wire together'", 'info')

        # Fetch credit on startup if logged in
        if self.config.authenticator:
            self.log("Fetching credit from server...", 'info')
            threading.Thread(target=lambda: self._fetch_credit_background(log_result=True), daemon=True).start()

        # Auto-resume if was running before
        if self.config.streaming.was_running and self.config.authenticator:
            self.log("Resuming training session...", 'info')
            self.root.after(1000, self.start_training)  # Delay slightly for UI to settle

        # Start schedule checker
        if self.config.streaming.schedule_enabled:
            self.log(f"Schedule active: {self._format_schedule()}", 'info')
        self.root.after(5000, self._schedule_check_loop)  # Start checking after 5 seconds

        self.root.mainloop()


if __name__ == "__main__":
    _log_startup("Creating AxiomStreamingGUI instance...")
    try:
        app = AxiomStreamingGUI()
        _log_startup("AxiomStreamingGUI created, calling run()...")
        app.run()
        _log_startup("run() returned normally")
    except Exception as e:
        _log_startup(f"FATAL ERROR: {e}\n{traceback.format_exc()}")
        raise
